% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program RSVP.m

function RSVP


%% Display Setup Module
% Define display parameters
whichScreen = max(Screen('screens'));
p.ScreenDistance = 30;  % in inches
p.ScreenHeight = 15;    % in inches
p.ScreenGamma = 2;  % from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5; 

% Open the display window, set up lookup table, and hide the 
% mouse cursor
if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end  
        % close any pre-existing PTB Screen window
% Prepare setup of imaging pipeline for onscreen window. 
PsychImaging('PrepareConfiguration'); % First step in starting
                                      % pipeline
PsychImaging('AddTask', 'General',  'FloatingPoint32BitIfPossible');  
        % set up a 32-bit floatingpoint framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange');   
        % normalize the color range ([0, 1] corresponds 
        % to [min, max])
PsychImaging('AddTask', 'General', 'EnablePseudoGrayOutput'); 
        % enable high gray level resolution output with 
        % bitstealing
PsychImaging('AddTask','FinalFormatting',  'DisplayColorCorrection','SimpleGamma');  
        % setup Gamma correction method using simple power 
        % function for all color channels 
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  
        % Finishes the setup phase for imaging pipeline
        % creates an onscreen window, performs all remaining  
        % configuration steps
PsychColorCorrection('SetEncodingGamma', windowPtr, 1/ p.ScreenGamma);  
% set Gamma for all color channels
HideCursor;  % Hide the mouse cursor 

% Get frame rate and set screen font
p.ScreenFrameRate = FrameRate(windowPtr); 
        % get current frame rate
Screen('TextFont', windowPtr, 'Times'); 
        % set the font for the screen to Times
Screen('TextSize', windowPtr, 24); % set the font size 
                                   % for the screen to 24


%% Experimental Module

% setup general experimental parameters
nTrials = 10;                % total number of trials
keys = [cellstr(char(65:90)')' {'delete' 'backspace' ...
       'enter' 'return'}];
p.randSeed = ClockRandSeed;  % use the clock to set random 
                             % number generator

% Specify the stimulus
p.textSize = 2.5;   % letter size in degrees of visual angle
p.letters = 'ABCDEFGHJKLMNOPRSTUWXYZ'; 
                    % subset of upper case letters
p.stimDuration = 0.05;   % duration of each letter in seconds
p.blankDuration = 0.1;   % duration of the blank interval between letters in seconds
p.fixDuration = 0.15;    % fixation duration in seconds 
p.iCueRange =[6 15];     % temporal range of the cue frame in 
                         % letter screen counts
p.textFont = 'Consolas'; % select a fixed width font for 
                         % stimulus display

% Compute stimulus parameters
halfInvt = 1 / p.ScreenFrameRate / 2; 
stimDur = round(p.stimDuration * p.ScreenFrameRate) ...
          / p.ScreenFrameRate - halfInvt;
blankDur = round(p.blankDuration * p.ScreenFrameRate) ...
          / p.ScreenFrameRate - halfInvt;
ppd = pi / 180 * p.ScreenDistance / p.ScreenHeight ...
          * p.ScreenRect(4);   % pixels/degree
textSize = round(p.textSize * ppd);
[xc yc] = RectCenter(p.ScreenRect);
fixLen = round(textSize / 4); % fixation size in pixels
fixXY = [[0 0 -1 1] * fixLen + xc;
         [-1 1 0 0] * fixLen + yc] ; % fixation xy
Screen('TextFont', windowPtr, p.textFont);
Screen('TextSize', windowPtr, textSize);
txtSz = Screen('TextBounds', windowPtr, 'A');
x1 = xc - txtSz(3) / 2; y1 = yc - txtSz(4) / 2; % letter xy
txtSz(3) = txtSz(3) * 1.5; % expand a bit to allow more room
cueRect = CenterRect(txtSz, p.ScreenRect);
 
Screen('TextSize', windowPtr, 48);
txtSz = Screen('TextBounds', windowPtr, 'ABCDEF');
feedbackRect = CenterRect(txtSz, p.ScreenRect);
feedbackRect(3) = feedbackRect(1) + txtSz(3) / 6;
x2 = feedbackRect(1); y2 = feedbackRect(2); 
nLetters = length(p.letters);
p.seq = repmat(p.letters', 1, nTrials);
p.seq = char(Shuffle(p.seq)); % shuffle letter stream for 
                              % each trial to randomize

% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'iCue' 'respTime' 'ans1' ...
              'ans2' 'ans3' 'ans4'};
rec = nan(nTrials, length(p.recLabel)); 
        % matrix rec is nTrials x 7 of NaN
rec(:, 1) = 1 : nTrials;    
        % assign trial numbers from 1 to nTrials
rec(:, 2) = randi(p.iCueRange, nTrials, 1); 
        % cue index for each trial
 
% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));
 
% Start experiment with instructions
str = ['Input 4 letters starting from the cued letter.\n\n' ...
        'Press Backspace or Delete to remove the last' ... 
        'input.\n\n' 'Press Enter or Return to finish your' ...
        'input.\n\n\n' 'Press SPACE to start.'];
Screen('TextSize', windowPtr, 36);
Screen('TextFont', windowPtr, 'Times'); 
DrawFormattedText(windowPtr, str, 'center', 'center', 1);
        % Draw Instruction text string centered in window
Screen('Flip', windowPtr);  
        % flip the text image into active buffer
WaitTill('space');  % wait till space bar is pressed
Screen('Flip', windowPtr);  % turn off instruction
p.start = datestr(now); % record start time
 
% Run nTrials trials
for i = 1 : nTrials
    Screen('DrawLines', windowPtr, fixXY, 3, 1);
    t0 = Screen('Flip', windowPtr); % show fixation
    t0 = Screen('Flip', windowPtr, t0 + p.fixDuration); 
        % turn off fixation
    
    Screen('TextSize', windowPtr, textSize);
    Screen('TextFont', windowPtr, p.textFont);
    for j = 1 : nLetters
        if j == rec(i, 2)
            Screen('FrameRect', windowPtr, 1, cueRect, 3);
        end
        Screen('DrawText', windowPtr, p.seq(j, i), x1, y1, ...
              1);
        t0 = Screen('Flip', windowPtr, t0 + blankDur); 
             % letter on
        t0 = Screen('Flip', windowPtr, t0 + stimDur); 
             % letter off
    end
    Screen('TextSize', windowPtr, 48);
    Screen('TextFont', windowPtr, 'Times'); 
    DrawFormattedText(windowPtr, 'Please input the 4 letters', 'center', 'center', 1);
    t0 = Screen('Flip', windowPtr); % show prompt string
    
    Screen('TextFont', windowPtr, p.textFont);
    ans4 = '';
    while 1   % collect 4 responses from keyboard and 
              % allow changes until return
        KbReleaseWait;
        [key Secs] = WaitTill(keys); 
        if iscellstr(key), key = key{1}; end 
 
        if any(strcmp(key, {'enter' 'return'})) % done
            if length(ans4) >= 4, break; end
        elseif any(strcmp(key, { 'delete' 'backspace'})) 
            if ~isempty(ans4), ans4(end) = []; end
        else
            ans4 = [ans4 upper(key)]; 
              % append input as upper case to feedback string
        end
        
        Screen('DrawText', windowPtr, ans4, x2, y2 - 80, 1);
        Screen('Flip', windowPtr); % show input
    end
    rec(i, 3) = Secs - t0; % record response time
    rec(i, 4 : 7) = ans4(1 : 4); % record the first 4 input
 
    Screen('DrawText', windowPtr, ans4, x2, y2 - 80, 1);
    Screen('DrawText', windowPtr, p.seq((0 : 5) + ...
          rec(i, 2), i)', x2, y2, 1);
    Screen('FrameRect', windowPtr, 1, feedbackRect, 1);
    Screen('TextFont', windowPtr, 'Times');
    DrawFormattedText(windowPtr, 'Press SPACE to proceed', ...
         'center', y2 + 120, 1);
    Screen('Flip', windowPtr); % show instruction
    if strcmp(WaitTill({'space' 'esc'}), 'esc'), break; end
end
 
p.finish = datestr(now); % record start time
 
save RSVP_rst rec p;     % save the results



%% System Reinstatement Module

Priority(0);  % restore priority
sca; % close window and textures, restore color lookup table

 